document.addEventListener('DOMContentLoaded', () => {
    // Referência atualizada: textInput agora é codeEditor
    const codeEditor = document.getElementById('code-editor');
    const fileInput = document.getElementById('file-input');
    const btnOpen = document.getElementById('btn-open');
    const btnNew = document.getElementById('btn-new');
    const btnSave = document.getElementById('btn-save');
    const btnSaveAs = document.getElementById('btn-save-as');
    const statusBar = document.getElementById('save-status');
    const currentFileEl = document.getElementById('current-file');
    const charCountEl = document.getElementById('char-count');
    const wordCountEl = document.getElementById('word-count');
    const lineCountEl = document.getElementById('line-count');

    // Estado do Editor
    let currentFileName = 'documento_sem_titulo.txt';
    let isSyntaxUpdating = false; // Flag para evitar loops e problemas de foco

    // Palavras-chave unificadas de Java, C/C++, Python (Exemplo)
    const KEYWORDS = [
        'if', 'else', 'for', 'while', 'return', 'class', 'public', 'private',
        'int', 'char', 'float', 'double', 'void', 'def', 'print', 'import',
        'try', 'catch', 'finally', 'new', 'this', 'static', 'const', 'true', 'false', 'null',
        'System', 'out', 'main', 'string', 'do', 'break', 'case' // Mais algumas para dar corpo
    ];
    // Regex para procurar palavras completas
    const keywordRegex = new RegExp(`\\b(${KEYWORDS.join('|')})\\b`, 'g');


    // FUNÇÃO CENTRAL PARA O REALCE DE SINTAXE
    // Aviso: Essa função é uma implementação básica e pode ter problemas de cursor
    const applySyntaxHighlighting = () => {
        if (isSyntaxUpdating) return;
        isSyntaxUpdating = true;

        // 1. Obtém o texto simples (textContent)
        const text = codeEditor.textContent;
        
        // 2. Armazena a posição atual do cursor (tentativa de manter o foco)
        const selection = window.getSelection();
        const range = selection.rangeCount > 0 ? selection.getRangeAt(0) : null;
        
        // 3. Substitui as keywords por versões estilizadas
        const highlightedText = text.replace(keywordRegex, '<span class="keyword">$&</span>');

        // 4. Aplica o HTML de volta no editor APENAS se houver alteração
        if (codeEditor.innerHTML !== highlightedText) {
            codeEditor.innerHTML = highlightedText;

            // 5. Tenta restaurar a posição do cursor (Parte mais complexa)
            if (range) {
                 // Define o ponto de foco no último nó de texto
                const lastChild = codeEditor.lastChild;
                const newRange = document.createRange();
                newRange.setStart(lastChild, lastChild.length);
                newRange.collapse(true);

                selection.removeAllRanges();
                selection.addRange(newRange);
            }
        }

        isSyntaxUpdating = false;
    };


    // Funções de Utilitário
    const updateStats = () => {
        // Usa textContent para obter o texto sem tags HTML
        const text = codeEditor.textContent;

        charCountEl.textContent = text.length;
        const words = text.match(/\b\w+\b/g);
        wordCountEl.textContent = words ? words.length : 0;
        const lines = text.split('\n');
        lineCountEl.textContent = lines.length;
        currentFileEl.textContent = currentFileName;
    };

    const setStatus = (message, isError = false) => {
        statusBar.textContent = message;
        statusBar.style.color = isError ? 'var(--error-color)' : 'var(--status-color)';
    };

    const saveFile = (filename) => {
        // Usa textContent para salvar apenas o texto puro
        const textContent = codeEditor.textContent;

        try {
            const blob = new Blob([textContent], { type: 'text/plain;charset=utf-8' });
            const url = URL.createObjectURL(blob);
            const a = document.createElement('a');
            a.href = url;
            a.download = filename;
            document.body.appendChild(a);
            a.click();
            document.body.removeChild(a);
            URL.revokeObjectURL(url);
            currentFileName = filename;
            updateStats();
            setStatus(`Arquivo salvo como "${filename}"!`);
        } catch (e) {
            setStatus('Erro ao salvar o arquivo.', true);
            console.error(e);
        }
    };


    // Listeners de Eventos (Atualizados para codeEditor)

    btnSave.addEventListener('click', () => {
        saveFile(currentFileName);
    });

    btnSaveAs.addEventListener('click', () => {
        let newFileName = prompt("Salvar arquivo como:", currentFileName);
        if (!newFileName) {
            setStatus('Operação Salvar Como cancelada.', true);
            return;
        }
        if (!newFileName.toLowerCase().endsWith('.txt')) {
            newFileName += '.txt';
        }
        saveFile(newFileName);
    });

    btnOpen.addEventListener('click', () => {
        fileInput.click();
    });

    fileInput.addEventListener('change', (event) => {
        const file = event.target.files[0];
        if (!file) return;

        const reader = new FileReader();

        reader.onload = (e) => {
            // Aplica o texto puro no editor
            codeEditor.textContent = e.target.result;
            currentFileName = file.name;
            applySyntaxHighlighting(); // Aplica o realce ao carregar
            updateStats();
            setStatus(`Arquivo "${file.name}" aberto.`);
        };

        reader.onerror = () => {
            setStatus('Erro ao ler o arquivo.', true);
        };

        reader.readAsText(file);
    });

    btnNew.addEventListener('click', () => {
        // Usa textContent para checar o valor
        if (codeEditor.textContent.trim() !== "" && !confirm("O conteúdo atual não foi salvo. Deseja iniciar um novo arquivo?")) {
            return;
        }
        codeEditor.innerHTML = ''; // Usa innerHTML para resetar o conteúdo
        currentFileName = 'documento_sem_titulo.txt';
        updateStats();
        setStatus('Pronto');
        codeEditor.focus();
    });

    // EVENTO CHAVE: Monitora a entrada do usuário para realçar
    codeEditor.addEventListener('input', () => {
        updateStats();
        setStatus('Editando...');
        // Chama o realce com um pequeno atraso (debounce)
        setTimeout(applySyntaxHighlighting, 50);
    });

    // Inicialização
    updateStats();
    codeEditor.focus();
});